//
//  ArrayList.h
//
//  Created by Anders Riggelsen on 6/21/13.
//  Copyright (c) 2013 Clickteam. All rights reserved.
//

#pragma once
#include <stdlib.h>
#include <cstdlib>
#include "CoreMath.h"
#include <string.h>

template <class T> class ArrayList
{
private:
	T* elements;
	size_t m_capacity;
	size_t numElements;

	void expandIfNecessaryForInsertion();
	void shrinkIfNecessary();

public:
	ArrayList();
	ArrayList(size_t capacity);
	~ArrayList();

	size_t size();								//Returns the size of the list (number of items in it)
	size_t capacity();							//Returns the current capacity of the list
	void ensureCapacity(size_t capacity);		//Expands the list to contain space for at least <capacity> elements
	void growAndFill(T &var, size_t count);		//Fills the array to a maximum of 'count' copies of 'var'.
	void crop(size_t startIndex, size_t size);	//Crops the array to the given size from the given start index
	void crop();								//Crops the array to use only the memory needed to contain the elements in the list
	size_t add(T &var);							//Insert at the end of the list and return the index
	size_t insert(T &var, size_t index);		//Insert at the given index
	void deleteIndex(size_t index);				//Deletes a given indexed node in the list
	void deleteElement(T &var);					//Deletes a node with the given data

	void clear();								//Clears the entire array
	void clearRetainingCapacity();				//Clears the entire array but keeps the memory for later allocation

	T &get(size_t index);						//Returns a given indexed element in the list
	T pop();

	T& operator[] (const size_t index);
};

template <class T> ArrayList<T>::ArrayList()
{
	this->m_capacity = 16;
	this->numElements = 0;
	this->elements = (T*)calloc(m_capacity, sizeof(T));
}

template <class T> ArrayList<T>::ArrayList(size_t capacity)
{
	this->m_capacity = capacity;
	this->numElements = 0;
	this->elements = (T*)calloc(capacity, sizeof(T));
}

template <class T> T& ArrayList<T>::operator[] (const size_t index)
{
	return elements[index];
}

template <class T> size_t ArrayList<T>::size()
{
	return numElements;
}

template <class T> size_t ArrayList<T>::capacity()
{
	return m_capacity;
}

template <class T> void ArrayList<T>::ensureCapacity(size_t capacity)
{
	if(m_capacity >= capacity)
		return;
	size_t extraspace = capacity - m_capacity;

	elements = (T*)realloc(elements, capacity*sizeof(T));
	memset(&this->elements[m_capacity], 0, extraspace*sizeof(T));
	m_capacity = capacity;
}

template <class T> void ArrayList<T>::growAndFill(T &var, size_t count)
{
	//If the array already has the elements necessary
	if(numElements >= count)
		return;

	ensureCapacity(count);
	size_t numExtra = count - numElements;
	for(int i=0; i<numExtra; ++i)
		add(var);
}

template <class T> void ArrayList<T>::expandIfNecessaryForInsertion()
{
	if(m_capacity == numElements)
	{
		size_t newSize = MIN(m_capacity*2, m_capacity+1024);	//Limit the size expansion to 1024 elements
		ensureCapacity(newSize);
	}
}

template <class T> void ArrayList<T>::shrinkIfNecessary()
{
	//Shrink if the usage is only 1/4 of the capacity (don't waste memory)
	size_t oneFourth = m_capacity/4;
	if(m_capacity > 16 && numElements < oneFourth)
	{
		elements = (T*)realloc(elements, oneFourth*sizeof(T));
		m_capacity = oneFourth;
	}
}

template <class T> void ArrayList<T>::crop(size_t startIndex, size_t size)
{
	//If none of the original elements will be in the new list
	if(startIndex > numElements)
	{
		if(size == numElements)
		{
			memset(elements, 0, numElements*sizeof(T));
		}
		else
		{
			free(elements);
			elements = (T*)calloc(size, sizeof(T));
		}
		return;
	}

	//Move the entire array from startIndex back to the beginning of the array
	if(startIndex != 0)
		memmove(&elements[0], &elements[startIndex], size*sizeof(T));

	//Resize
	elements = (T*)realloc(elements, size*sizeof(T));

	//Zero out the new elements if the new size is larger than the old one
	size_t oldLastIndex = (size_t)MAX((int)numElements-(int)startIndex, 0);
	size_t numNewElements = (size_t)MAX((int)size-(int)numElements, 0) + startIndex;

	if(numNewElements > 0)
		memset(&elements[oldLastIndex], 0, numNewElements*sizeof(T));

	m_capacity = size;
	numElements = size;
}

template <class T> void ArrayList<T>::crop()
{
	crop(0, numElements);
}

template <class T> size_t ArrayList<T>::add(T &var)
{
	expandIfNecessaryForInsertion();
	elements[numElements] = var;
	return numElements++;
}

template <class T> size_t ArrayList<T>::insert(T &var, size_t index)
{
	expandIfNecessaryForInsertion();

	if(index < numElements)
		memmove(&elements[index], &elements[index+1], (numElements-index)*sizeof(T));	//Move the rest of the items
	else
		return add(var);	//insert at the end of the list out of bounds

	elements[index] = var;
	return index;
}

template <class T> T &ArrayList<T>::get(size_t index)
{
	assert(index < numElements);
	return elements[index];
}

template <class T> T ArrayList<T>::pop()
{
	T ret = elements[numElements-1];
	if(numElements > 0)
	{
		memset(&elements[numElements-1], 0, sizeof(T));
		numElements--;
	}
	shrinkIfNecessary();
	return ret;
}

template <class T> void ArrayList<T>::deleteIndex(size_t index)
{
	if(index < numElements-1)
		memmove(&elements[index], &elements[index+1], (numElements-index)*sizeof(T));
	memset(&elements[numElements], 0, sizeof(T));
	--numElements;
	shrinkIfNecessary();
}

template <class T> void ArrayList<T>::deleteElement(T &var)
{
	for(int i=0; i<numElements; ++i)
	{
		if(elements[i] == var)
		{
			deleteIndex(i);
			return;
		}
	}
}

template <class T> void ArrayList<T>::clear()
{
	numElements = 0;
	m_capacity = 16;
	free(elements);
	elements = (T*)calloc(capacity, sizeof(T));
}

template <class T> void ArrayList<T>::clearRetainingCapacity()
{
	numElements = 0;
	memset(elements, 0, sizeof(T)*m_capacity);
}

template <class T> ArrayList<T>::~ArrayList()
{
	free(elements);
	numElements = 0;
	m_capacity = 0;
	elements = NULL;
}
