//
//  CRuniOSNotif.m
//  RuntimeIPhone
//
//  Created by Timothy Ellis on 19/05/12.
//  Copyright (c) 2012 Tim Ellis. All rights reserved.
//
//  www:    http://insaynity.net
//  email:  tim.ellis@insaynity.net
//
//  Version: 1.1 Stable

#import "CRuniOSNotif.h"

#import "CExtension.h"
#import "CRun.h"
#import "CRunApp.h"
#import "CBitmap.h"
#import "CServices.h"
#import "CCreateObjectInfo.h"
#import "CImage.h"
#import "CActExtension.h"
#import "CCndExtension.h"
#import "CValue.h"

#import <UIKit/UIKit.h>
#import "RuntimeIPhoneAppDelegate.h"

#define CND_COUNT 4

#pragma mark - Private Variables
@interface CRuniOSNotif ()

@property NSUInteger componentFlags;
@property NSInteger day;
@property NSInteger month;
@property NSInteger year;
@property NSInteger hour;
@property NSInteger minute;
@property NSInteger second;

@property (nonatomic, assign) NSString *localSoundPath;
@property (nonatomic, assign) NSString *bodyContents;
@property (nonatomic, retain) NSError *onError;
@property (nonatomic, retain) NSData *token;
@property (nonatomic, retain) NSDictionary *notificationData;
@property (nonatomic, retain) UILocalNotification *localNotif;

@end

#pragma mark - Extension
@implementation CRuniOSNotif

@synthesize componentFlags;
@synthesize day, month, year, hour, minute, second;

@synthesize localSoundPath;
@synthesize bodyContents;
@synthesize onError;

@synthesize token;
@synthesize notificationData;
@synthesize localNotif;

#pragma mark - iOS Notifications Handler
- (void)application:(UIApplication*)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData*)deviceToken
{
    token = deviceToken;
    //Call On Registered For Remote Notifications
    [ho generateEvent:1 withParam:0];
}

- (void)application:(UIApplication*)application didFailToRegisterForRemoteNotificationsWithError:(NSError*)error
{
	NSLog(@"Failed to get token, error: %@", [error localizedDescription]);
    onError = error;
    [ho generateEvent:2 withParam:0];
}

- (void)application:(UIApplication*)application didReceiveRemoteNotification:(NSDictionary*)userInfo
{
    notificationData = userInfo;
    [ho generateEvent:0 withParam:0];
}

- (void)application:(UIApplication *)application didReceiveLocalNotification:(UILocalNotification *)notification
{
    localNotif = notification;
    [ho generateEvent:3 withParam:0];
}

#pragma mark - Actions
-(void) registerForRemoteNotificationsWithBadge:(BOOL)badge Sound:(BOOL)sound Alert:(BOOL)alert{
    
    UIRemoteNotificationType notif = UIRemoteNotificationTypeNone;
    if (badge)
        notif = UIRemoteNotificationTypeBadge;
    if (sound)
        notif |= UIRemoteNotificationTypeSound;
    if (alert)
        notif |= UIRemoteNotificationTypeAlert;
    
    [[UIApplication sharedApplication] registerForRemoteNotificationTypes: notif];
}

-(void) scheduleLocalNotification{
    
    NSDateComponents *comps = [[NSDateComponents alloc] init];
    
    [comps setDay:day];
    [comps setMonth:month];
    [comps setYear:year];
    
    [comps setHour:hour];
    [comps setMinute:minute];
    [comps setSecond:second];
    
    NSCalendar *cal = [[NSCalendar alloc] initWithCalendarIdentifier:NSGregorianCalendar];
    NSDate *date = [cal dateFromComponents:comps];
    [comps release];
    
    UIApplication* app = [UIApplication sharedApplication];
    UILocalNotification* notifyAlarm = [[UILocalNotification alloc]
                                        init];
    if (notifyAlarm)
    {
        notifyAlarm.fireDate = date;
        notifyAlarm.timeZone = [NSTimeZone defaultTimeZone];
        notifyAlarm.repeatInterval = 0;
        notifyAlarm.soundName = localSoundPath;
        notifyAlarm.alertBody = bodyContents;
        [app scheduleLocalNotification:notifyAlarm];
    }
}

-(void) clearNotifications{
    
    [[UIApplication sharedApplication] cancelAllLocalNotifications];
    
}

-(void) setApplicationBadgeNumber:(NSInteger)badge{
    
    [[UIApplication sharedApplication] setApplicationIconBadgeNumber:badge];
    
}

-(void) clearApplicationBadge{
    
    //Simply changing the badge to zero doesn't always work
    //Forcing a change will work
    [[UIApplication sharedApplication] setApplicationIconBadgeNumber:1];
    //Zero means no badge
    [[UIApplication sharedApplication] setApplicationIconBadgeNumber:0];
    
}

#pragma mark - MMF2 Code

//This function should return the number of conditions contained in the object (equivalent to the CND_LAST define in the ext.h file).
-(int)getNumberOfConditions
{
    return CND_COUNT;
}

/*
 This function is called when the object is created. As iOS object are just created when needed, there is no EDITDATA structure.
 Instead, I send to the function a CFile object, pointing directly to the data of the object (the EDITDATA structure, on disc).
 The CFile object allows you to read the data. It automatically converts PC-like ordering (big or little Indian I cant remember) into iOS ordering.
 It contains functions to read bytes, shorts, int, colors and strings.
 
 So all you have to do, is read the data from the CFile object, and initialise your object accordingly.
 Return YES if your object has been created successfully.
 */
-(BOOL)createRunObject:(CFile *)file withCOB:(CCreateObjectInfo *)cob andVersion: (int)version
{
    NSDate *today = [NSDate date];
    componentFlags = (NSYearCalendarUnit | NSMonthCalendarUnit | NSDayCalendarUnit | NSHourCalendarUnit | NSMinuteCalendarUnit | NSSecondCalendarUnit);
    
    NSDateComponents *todayDayComponents = [[NSCalendar currentCalendar] components:componentFlags fromDate:today];
    
    day = [todayDayComponents day];
    month = [todayDayComponents month];
    year = [todayDayComponents year];
    
    hour = [todayDayComponents hour];
    minute = [todayDayComponents minute];
    second = [todayDayComponents second];
    
    localSoundPath = @"";
    onError = nil;
    
    [rh->rhApp registerForiOSEvents:self];
    
    [self registerForRemoteNotificationsWithBadge:YES Sound:YES Alert:YES];
    
    return YES;
}

/*
 Called when the object is destroyed. This routine should free all the memory allocated during createRunObject.
 bFast is true if the object is destroyed at end of frame. It is false if the object is destroyed in the middle of the application.
 */
-(void)destroyRunObject:(BOOL)bFast
{
    [token release];
    [notificationData release];
    [localNotif release];
    [onError release];
    [rh->rhApp unregisterForiOSEvents:self];
}

/*
 Same as the C++ function. Perform all the tasks needed for your object in this function.
 As the C function, this function returns value indicating what to do :
 • REFLAG_ONESHOT : handleRunObject will not be called anymore
 • REFLAG_DISPLAY : displayRunObject is called at next refresh.
 • Return 0 and the handleRunObject method will be called at the next loop.
 */
-(int)handleRunObject
{
    return REFLAG_ONESHOT;
}

//Called when the application goes into pause mode.
-(void)pauseRunObject
{
    
}

//Called when the application restarts.
-(void)continueRunObject
{
    
}

/*
 The main entry for the evaluation of the conditions.
 • num : number of the condition (equivalent to the CND_ definitions in ext.h)
 • cnd : a pointer to a CCndExtension object that contains useful callback functions to get the parameters.
 This function should return YES or NO, depending on the condition.
 */
-(BOOL)condition:(int)num withCndExtension:(CCndExtension *)cnd
{
    return YES;
}

/*
 The main entry for the actions.
 • num : number of the action, as defined in ext.h
 • act : pointer to a CActExtension object that contains callback functions to get the parameters.
 */
-(void)action:(int)num withActExtension:(CActExtension *)act
{
    NSDate *today;
    NSDateComponents *todayDayComponents;
    //NSString *tmp;
    
    switch (num)
    {
        case 0:
            //Clear All Local Notifications
            [self clearNotifications];
            break;
            
        case 1:
            //Set Fire Date & Time
            day = [act getParamExpression:rh withNum:0];
            month = [act getParamExpression:rh withNum:1];
            year = [act getParamExpression:rh withNum:2];
            hour = [act getParamExpression:rh withNum:3];
            minute = [act getParamExpression:rh withNum:4];
            second = [act getParamExpression:rh withNum:5];
            
            break;
            
        case 2:
            //Set Trigger (Seconds in the Future)
            today = [[NSDate date] dateByAddingTimeInterval: [act getParamExpression:rh withNum:0]];
            todayDayComponents = [[NSCalendar currentCalendar] components:componentFlags fromDate:today];
            
            day = [todayDayComponents day];
            month = [todayDayComponents month];
            year = [todayDayComponents year];
            
            hour = [todayDayComponents hour];
            minute = [todayDayComponents minute];
            second = [todayDayComponents second];
            break;
            
        case 3:
            //Set Body Contents
            bodyContents = [act getParamExpString:rh withNum:0];
            break;
        
        case 4:
            //Schedule Notification
            [self scheduleLocalNotification];
            break;
            
        case 5:
            //Set Application Badge
            [self setApplicationBadgeNumber:[act getParamExpression:rh withNum:0]];
            break;
            
        case 6:
            //Clear Application Badge
            [self clearApplicationBadge];
            break;
        
        case 7:
            //Set Notification sound
            localSoundPath = [act getParamExpString:rh withNum:0];
            //tmp = [tmp lastPathComponent];
            /*
            if ([[NSFileManager defaultManager] fileExistsAtPath:tmp]){
                localSoundPath = tmp;
                localSoundPath = [[NSBundle mainBundle] pathForResource:[localSoundPath stringByDeletingPathExtension] ofType:[localSoundPath pathExtension]];
            }
            else
                localSoundPath = @"";
            */
            break;
    }
}

/*
 The main entry for expressions.
 • num : number of the expression
 To get the expression parameters, you have to call the getExpParam method defined in the "ho" variable, for each of the parameters.
 This function returns a CValue* which contains the parameter.
 You then do a getInt(), getDouble() or getString() with the CValue object to grab the actual value.
 This function returns a pointer to a Cvalue object containing the result.
 The content of the CValue can be a integer, a double or a String.
 There is no need to set the HOF_STRING flags if your return a string : the CValue object contains the type of the returned value.
 You should not alloc the CValue yourself, but instead ask for a temporary value from the Crun class :
 [rh getTempValue:default_value] where default_value is an integer.
 */

-(CValue *)expression:(int)num
{
    NSString *tempStr = @"";
    switch (num)
    {
            //Get Device Token
        case 0:
            tempStr = [[token description] stringByTrimmingCharactersInSet:[NSCharacterSet characterSetWithCharactersInString:@"<>"]];
            tempStr = [[tempStr componentsSeparatedByString:@" "] componentsJoinedByString:@""];
            return [rh getTempString:tempStr];
            //Get Device Token (RAW)
        case 1:
            return [rh getTempString:[token description]];
            //Get Notification -> Message
        case 2:
            if (notificationData)
                return [rh getTempString:[[notificationData objectForKey:@"aps"] objectForKey:@"alert"]];
            return [rh getTempString:@""];
            //Get Notification -> Custom Token
        case 3:
            //Get the custom token
            if (notificationData){
                tempStr = [[ho getExpParam] getString];
                return [rh getTempString:[[notificationData objectForKey:@"aps"] objectForKey:tempStr]];
            }
            return [rh getTempString:@""];
        case 4:
            //Get Error Reason
            if (onError){
                tempStr = [onError localizedDescription];
                return [rh getTempString:tempStr];
            }
            else
                return [rh getTempString:@""];
        case 5:
            //Get Local Notification -> Message
            if (localNotif == nil)
                return [rh getTempString:@"nil"];
            tempStr = [localNotif alertBody];
            return [rh getTempString:tempStr];
    }
    return [rh getTempValue:0];
}


@end
